#ifndef AHLGREN_FUNTABLE
#define AHLGREN_FUNTABLE

#include <iostream>
#include <climits>
#include <string>
#include <cstring>
#include <map>
// #include <unordered_map>
#include <cassert>
#include <cstdlib> // malloc
#include <cstdio> // sprintf (for conversion int->Cstring)

#include "useful.h"

namespace lp {
	using namespace std;
	using namespace useful;

	// Store functor symbols in a map
	/*	value 0 indicates nil
		values [1,INT_MAX/2[ indicate system variable
		values [INT_MAX/2, INT_MAX] indicate user variable
		values [INT_MIN/2, -1] indicate system functor
		values [INT_MIN, INT_MIN/2[ indicate user functor
		i.e.:
		<-- user functors --- sys functors --- 0 --- sys vars --- user vars -->
	*/
	typedef int id_type; // type of IDs that functor works with

	// UPDATE: use unrestricted union when std::string can be embedded (C++11)
	class data {
	public:
		// static const int llsize = sizeof(long long);
		static const int int_tag = -1;
		static const int float_tag = -2;
		data() : tag(0) {}
		data(const char* ptr);
		data(const string& str);
		data(long long ii) : tag(int_tag), i(ii) { }
		data(double dd) : tag(float_tag), d(dd) { }
		data(const data&);
		data& operator=(const data&);
		data(data&&);
		~data() { if (is_atom()) free(s); }
		data& operator=(data&&);
		// Compare two elements, returns -1 : 0 : 1 for order, or -2 if different types
		int compare(const data&) const;
		bool operator==(const data& dat) const { return this->compare(dat) == 0; }
		bool operator!=(const data& dat) const { return !(*this == dat); }
		// Note: semantics is non-standard for orderings: two different types always give false
		bool operator<(const data&) const;
		bool operator>(const data& dat) const;
		bool operator<=(const data&) const;
		bool operator>=(const data&) const;
		// Check types
		bool is_nil() const { return tag == 0; }
		bool is_atom() const { return tag > 0; }
		bool is_int() const { return tag == data::int_tag; }
		bool is_double() const { return tag == data::float_tag; } 
		bool is_numeric() const { switch (tag) { case data::int_tag: case data::float_tag: return true; } return false; } 
		// Get type
		int type() const { return tag; }
		long long get_int() const { return i; }
		double get_float() const { return d; }
		const char* get_atom() const { return s; }
		// Set value
		void set_int(long long val) { assert(is_int()); i = val; }
		void set_float(double val) { assert(is_double()); d = val; }
		// modifying atom value not permitted
	protected:
		// Data Members
		int tag; // tag: positive=strlen(s)+1, nil=0, int=-1, double=-2
		// Note: small string optimization hurts performance, as it
		// inhibits move operations with pointers
		// bool is_dynamic_atom() const { return tag > llsize; }
		// bool is_static_atom() const { return tag > 0 && tag <= llsize; }
		union {
			char* s;
			double d;
			long long i;
			// char ss[llsize];
		};
	};

	// Compare data (for map)
	struct data_cmp {
		bool operator()(const data& x, const data& y) const {
			if (x.type() < y.type()) return true;
			if (x.type() > y.type()) return false;
			// (x.tag == y.tag) => same type
			switch (x.type()) {
			case 0: return false; // same
			case data::int_tag: return x.get_int() < y.get_int();
			case data::float_tag: return x.get_float() < y.get_float();
			default: return strcmp(x.get_atom(),y.get_atom()) < 0;
			}
		}
	};

	ostream& operator<<(ostream& os, const data& d);
	int numeric_compare(const data& d1, const data& d2);
	// Helpers
	//template <typename T> bool operator==(const data& dat, T v)  { return dat == data(v); }
	//bool operator!=(const data& dat, long long v);
	//bool operator<(const data& dat, long long v);
	//bool operator<=(const data& dat, long long v);
	//bool operator>(const data& dat, long long v);
	//bool operator>=(const data& dat, long long v);
	//inline bool operator<(const data& dat, long long v)
	//{
	//	if (!dat.is_int()) return false;
	//	return dat.get_int() < v;
	//}


	// Define symbols that correspond to built-in predicates
	// UPDATE: use numeric_limits when constexpr is supported
	const id_type pair_id = INT_MIN/2 + 1; // .
	const id_type empty_list_id = INT_MIN/2 + 2; // []
	const id_type sequence_id = INT_MIN/2 + 3; // ,
	const id_type if_id = INT_MIN/2 + 4; // :-
	const id_type plus_id = INT_MIN/2 + 5; // +
	const id_type minus_id = INT_MIN/2 + 6; // -
	const id_type mul_id = INT_MIN/2 + 7; // *
	const id_type div_id = INT_MIN/2 + 8; // /
	const id_type pow_id = INT_MIN/2 + 9; // ^
	const id_type hash_id = INT_MIN/2 + 10; // #
	const id_type cut_id = INT_MIN/2 + 11; // !
	const id_type naf_id = INT_MIN/2 + 12; // \+
	const id_type not_id = INT_MIN/2 + 13; // not (equivalent to \+)
	const id_type or_id = INT_MIN/2 + 14; // ;
	const id_type query_id = INT_MIN/2 + 15; // ?-
	const id_type equal_id = INT_MIN/2 + 16; // =
	const id_type unequal_id = INT_MIN/2 + 17; // \=
	const id_type syn_equal_id = INT_MIN/2 + 18; // ==
	const id_type syn_unequal_id = INT_MIN/2 + 19; // \==
	const id_type is_id = INT_MIN/2 + 20; // is
	const id_type math_equal_id = INT_MIN/2 + 21; // =:=
	const id_type math_unequal_id = INT_MIN/2 + 22; // =:=
	const id_type lt_id = INT_MIN/2 + 23; // <
	const id_type gt_id = INT_MIN/2 + 24; // >
	const id_type le_id = INT_MIN/2 + 25; // =<
	const id_type ge_id = INT_MIN/2 + 26; // >=
	const id_type univ_id = INT_MIN/2 + 27; // =..
	const id_type lp_id = INT_MIN/2 + 28; // (
	const id_type rp_id = INT_MIN/2 + 29; // )
	const id_type ll_id = INT_MIN/2 + 30; // [
	const id_type rl_id = INT_MIN/2 + 31; // ]
	const id_type lsep_id = INT_MIN/2 + 32; // |
	const id_type zero_id = INT_MIN/2 + 33; // 0
	const id_type s_id = INT_MIN/2 + 34; // s
	const id_type any_id = INT_MIN/2 + 35; // any (type)
	const id_type true_id = INT_MIN/2 + 36; //
	const id_type fail_id = INT_MIN/2 + 37; //
	const id_type false_id = INT_MIN/2 + 38; //
	const id_type var_id = INT_MIN/2 + 39; //
	const id_type nonvar_id = INT_MIN/2 + 40; //
	const id_type atomic_id = INT_MIN/2 + 41; //
	const id_type number_id = INT_MIN/2 + 42; //
	const id_type functor_id = INT_MIN/2 + 43; //
	const id_type arg_id = INT_MIN/2 + 44; //
	const id_type set_id = INT_MIN/2 + 45; //
	const id_type halt_id = INT_MIN/2 + 46; //
	const id_type op_id = INT_MIN/2 + 47; //
	const id_type listing_id = INT_MIN/2 + 48; //
	const id_type display_id = INT_MIN/2 + 49; //
	const id_type consult_id = INT_MIN/2 + 50; //
	const id_type assert_id = INT_MIN/2 + 51; //
	const id_type asserta_id = INT_MIN/2 + 52; //
	const id_type assertz_id = INT_MIN/2 + 53; //
	const id_type retract_id = INT_MIN/2 + 54; //
	//const id_type compress_id = INT_MIN/2 + 55; //
	const id_type bottom_id = INT_MIN/2 + 56; //
	const id_type generalize_id = INT_MIN/2 + 57; //
	const id_type modeh_id = INT_MIN/2 + 58; //
	const id_type modeb_id = INT_MIN/2 + 59; //
	const id_type or2_id = INT_MIN/2 + 60; // builtin _; for RHS of OR in SLD resolution
	const id_type qmark_id = INT_MIN/2 + 61; // suffix '?' for Progol style queries
	const id_type generalise_id = INT_MIN/2 + 62; // brittish spelling for Progol users
	const id_type one_id = INT_MIN/2 + 63; // 1
	const id_type unset_id = INT_MIN/2 + 64; // unset
	const id_type get_id = INT_MIN/2 + 65; // get
	const id_type constant_id = INT_MIN/2 + 66; // constant (Progol syn for atomic)
	const id_type atom_id = INT_MIN/2 + 67;
	const id_type findall_id = INT_MIN/2 + 68;
	const id_type bagof_id = INT_MIN/2 + 69;
	const id_type bagof2_id = INT_MIN/2 + 70; // builtin _bagof for backtracking
	const id_type sort_id = INT_MIN/2 + 71;
	const id_type setof_id = INT_MIN/2 + 72;
	const id_type clause_id = INT_MIN/2 + 73;
	const id_type once_id = INT_MIN/2 + 74;
	const id_type integer_id = INT_MIN/2 + 75;
	const id_type float_id = INT_MIN/2 + 76;
	const id_type ifthen_id = INT_MIN/2 + 77;
	const id_type star2_id = INT_MIN/2 + 78; // **, for pow
	const id_type ifbranch_id = INT_MIN/2 + 79; // _->, internal branching used with ->
	const id_type write_id = INT_MIN/2 + 80;
	const id_type nl_id = INT_MIN/2 + 81;
	const id_type nrsample_id = INT_MIN/2 + 82; // induce: Atom's DPLL constraint satisfaction
	const id_type qg_id = INT_MIN/2 + 83; // induce: Progol's quick generalization
	const id_type qgga_id = INT_MIN/2 + 84; // induce: Progol's quick generalization
	const id_type functional_id = INT_MIN/2 + 85; // keyword "functional" for mode declarations
	const id_type equal2_id = INT_MIN/2 + 86; // _=, used by variable splitting
	const id_type sample_id = INT_MIN/2 + 87; // sample/1, stochastic query sampling
	const id_type label_id = INT_MIN/2 + 88; // label/2, clause label
	const id_type nat_id = INT_MIN/2 + 89; // nat/0, natural type for mode declaration
	const id_type vsids_id = INT_MIN/2 + 90; // VSIDS DPLL selection
	const id_type svsids_id = INT_MIN/2 + 91; // stochastic VSIDS DPLL selection
	const id_type random_id = INT_MIN/2 + 92; // random DPLL selection
	const id_type determination_id = INT_MIN/2 + 93; // determination/2
	const id_type evalfn_id = INT_MIN/2 + 94; // evalfn
	const id_type compression_id = INT_MIN/2 + 95; // evalfn: coverage
	const id_type posonly_id = INT_MIN/2 + 96; // evalfn: posonly
	const id_type minset_id = INT_MIN/2 + 97; // induce: Atom's 3stage algorithm
	const id_type heuristic_id = INT_MIN/2 + 98; // induce: Progol's best first search
	const id_type enumerate_id = INT_MIN/2 + 99; // induce: enumeration
	const id_type int_id = INT_MIN/2 + 100; // synonym for integer/1
	const id_type name_id = INT_MIN/2 + 101; // name/2
	const id_type retractall_id = INT_MIN/2 + 102; // retractall/1
	const id_type listing_wam_id = INT_MIN/2 + 103; // listing_wam/1
	const id_type call_id = INT_MIN/2 + 104; // call/1
	const id_type emulate_nrsample_id = INT_MIN/2 + 105;
	const id_type lessat_id = INT_MIN/2 + 106; // @<
	const id_type lesseqat_id = INT_MIN/2 + 107; // @=<
	const id_type greaterat_id = INT_MIN/2 + 108; // @>
	const id_type greatereqat_id = INT_MIN/2 + 109; // @>=
	const id_type compare_id = INT_MIN/2 + 110; // @>=
	const id_type isvariant_id = INT_MIN/2 + 111;
	const id_type isnotvariant_id = INT_MIN/2 + 112;
	const id_type nb_linkarg_id = INT_MIN/2 + 113;
	const id_type duplicate_term_id = INT_MIN/2 + 114;
	const id_type reflexive_id = INT_MIN/2 + 115; // keyword "reflexive" for mode declarations
	const id_type symmetric_id = INT_MIN/2 + 116; // keyword "symmetric" for mode declarations
	const id_type irreflexive_id = INT_MIN/2 + 117; // keyword "irreflexive" for mode declarations
	const id_type antisymmetric_id = INT_MIN/2 + 118; // keyword "antisymmetric" for mode declarations
	const id_type transitive_id = INT_MIN/2 + 119; // keyword "transitive" for mode declarations
	const id_type implies_id = INT_MIN/2 + 120; // '=>'/2 to declare relational implications
	const id_type idempotent_id = INT_MIN/2 + 121; // keyword "idempotent" for mode declarations
	const id_type prevents_id = INT_MIN/2 + 122; // prevents/2 to declare bottom clause blocking
	const id_type prevent_id = INT_MIN/2 + 123; // prevent/1 to declare bottom clause blocking
	const id_type id_nrsample_id = INT_MIN/2 + 124; // iterated deepening version of nrsample
	const id_type exclude_predicate_id = INT_MIN/2 + 125; // exclude predicate during induction (dual of determination/2)
	const id_type include_predicate_id = INT_MIN/2 + 126; // include predicate during induction (synonym for determination/2)

	class functor_map {
	public:
		functor_map();
		// note: no destructor, since data deletes its own memory
		static bool is_reserved_var(id_type i) { return i > 0 && i <= INT_MAX/2; }
		static bool is_reserved_fun(id_type i) { return i < 0 && i >= INT_MIN/2; }
		static bool is_reserved(id_type i) { return is_reserved_var(i) || is_reserved_fun(i); }
		static string make_varname(id_type i) { return "_G" + stringify(i); }
		static string make_funname(id_type i) { return "_f" + stringify(-i); }
		static bool is_variable(const string& s);
		id_type new_var_id() { return ++rvc; }
		id_type new_fun_id() { return --rfc; }

		// Get functor/integer/double
		data get_data(id_type i) const;
		// Get atom from ID (or throw not_found)
		const char* get_atom(id_type i) const;
		long long get_int(id_type i) const;
		double get_float(id_type i) const;
		data get_numeric(id_type i) const;

		// Get ID from string (create new id if needed)
		id_type id(const string& s);
		id_type id(const char* s);
		// Get ID from int (create new id if needed)
		id_type id(long long v);
		// Get ID from double (create new id if needed)
		id_type id(double v);
		// Get ID from data (create new id if needed)
		id_type id(const data&);
		id_type id(data&&);
		// Print id to ostream
		void print(ostream&, id_type) const;

		int size() const { return int(imap.size()); }
	private:
		// data -> id
		map<data,id_type,data_cmp> imap;
		// id -> data*
		map<id_type,const data*> dmap;
		id_type vc, fc, rvc, rfc;

		// helper to initialize functor_map with builtin predicates
		void add(id_type i, const char* s);
		void add(id_type i, long long v);
		void add(id_type i, double v);
	};

}


#endif

